% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program PhotometricTest.m

function PhotometricTest

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenGamma = 1;	 % set it to 1 so we can measure the real gamma
p.ScreenBackground = 0;

% Open display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
PsychImaging('PrepareConfiguration'); 
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % open a display window
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);
HideCursor;

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);


%% Experimental Module
%Specify the stimulus
bits = 4;        % determine the minimum step, could be 
                 % changed by the user
val = round(0.5 * 2 ^ bits) / 2 ^ bits; 
                 % initial gray value, may change
rgb = [1 1 1];   % channels to test; can be changed by
                 % the user using the r, g and b keys
keys = {'down' 'up' 'space' '+' '=' '-' 'r' 'g' 'b' ...
        'f1' 'esc'}; % Define control keys
 
% Help instruction to change parameters, and to turn 
% on/off instruction
hStr=['Up / Down arrow keys: increase / decrease value\n' ...
      '+ /  - : incease / decrease bits\n', ...
      'r g or b: turn a channel on / off\n' ...
      'F1: Show / hide this key help information\n', ...
      'ESC: Exit\n'];
showHelp = 1; % whether to show help text at beginning
showVals = 1; % whether to show values text at beginning
 
% Show the full screen for test, allow changing parameters
while 1
    Screen('FillRect', windowPtr, val * rgb); 
        % draw the value
    
    texColor = mod(val + 0.45, 1); 
        % make sure text is visible
    if showVals
        str = sprintf('Value = %.*f;', ...
              ceil(log10(2 ^ bits)), val);
        str = sprintf('%s  Bits = %g;', str, bits);
        str = sprintf('%s  RGB = [%g %g %g]', str, rgb);
        Screen('DrawText', windowPtr, str, 4, 4, texColor);
    end
    
    if showHelp
        DrawFormattedText(windowPtr, hStr, 4, 64, ...
              texColor, [], 0, 0, 1.5);
    end
    Screen('Flip', windowPtr);
    
    KbReleaseWait; % wait until all keys on the keyboard
                   % are released
    key = WaitTill(keys);      % wait for defined key press
 
    % Change parameters according to key press
    if     strcmp(key, 'down'),  val = val -2 ^ -bits;
    elseif strcmp(key, 'up'),    val = val +2 ^ -bits;
    elseif strcmp(key, 'space'), showVals = 1 - showVals;
    elseif strcmp(key, '+'),     bits = bits + 1;
    elseif strcmp(key, '='),     bits = bits + 1;
    elseif strcmp(key, '-'),     bits = bits - 1;
    elseif strcmp(key, 'r'),     rgb(1) = 1 - rgb(1);
    elseif strcmp(key, 'g'),     rgb(2) = 1 - rgb(2);
    elseif strcmp(key, 'b'),     rgb(3) = 1 - rgb(3);
    elseif strcmp(key, 'f1'),    showHelp = 1 - showHelp;
    elseif strcmp(key, 'esc'),   break;
    end
    
    val = round(val * 2 ^ bits) / 2 ^ bits;
    val = max(0, min(1, val));    % within 0 and 1
    bits = max(2, min(16, bits)); % 4 to 16 bits
end 


%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table


